/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */
package djbuch.kapitel_15; import java.util.Vector;

import djbuch.kapitel_06.GC;
import lotus.domino.*;

/**
 * Kleines DemoObjekt, das die 
 * verschiedenen Szenarien des Recyclings von Domino Objekten 
 * aufzeigt.
 * 
 * @author Thomas Ekert
 *
 */
public class ToyStore {

	private Database db = null;
	private Session session = null;
	private static final String ITEM_ID = "F_ID";
	public static final String ITEM_NAME = "F_NAME";	
	public static final String ITEM_FORM = "FORM";	
	public static final String DEFAULT_FORM = "TOYFORM";	
	private Vector trash = new Vector();
	
	/**
	 * Neues ToyStoreObejkt anlegen.
	 * @param dbName - Name der Datenbank. Wird als internes Datebank Objekt angelegt und geffnet
	 * @param session - Um mit Domino Objekten zu arbeiten, wird immer eine Session bentigt.
	 * @throws NotesException
	 */
	public ToyStore (String dbName, Session session) throws NotesException {
		db = session.getDatabase(session.getServerName(),dbName);
		if (db==null || !db.isOpen()) {
			throw new NotesException (999, "Konnte die Datenbank " + dbName + " nicht ffnen.");
		}
		this.session = session;
	}

	/**
	 * Ldt den Namen eines Toys mit der ID id
	 * Da das Document doc nur intern in dieser Methode und nur temporr
	 * bentigt wird, kann die Methode selbst das Recycle bernehmen.
	 * @param id - ID des zu ladenden Toys
	 * @return - Name des Toys oder null, falls nicht gefunden.
	 * @throws NotesException
	 */
	public String getToy (int id) throws NotesException {
		checkRecycled();
		Document doc = null;
		DocumentCollection dc = null;
		try {
			dc = db.search(ITEM_ID + "=" + id);
			if (dc==null || dc.getCount()==0) {
				return null;
			}
			doc = dc.getFirstDocument();
			return (doc==null?null:doc.getItemValueString(ITEM_NAME));
		} finally {
			GC.recycle (doc);
			GC.recycle (dc);
		}
	}
	
	/**
	 * Ldt das zu einem Toy gehrige Domino Document mit der ID id
	 * Da das Document doc als Ergebnis zurckgegeben wird,
	 * kann die Methode das Recycle nicht (!) selbst bernehmen.
	 * @param id - ID des zu ladenden Toys
	 * @return - Document mit dem Toy oder null, falls nicht gefunden.
	 * @throws NotesException
	 */
	public Document getToyDocument (int id) throws NotesException {
		checkRecycled();
		DocumentCollection dc = null;
		try {
			dc = db.search(ITEM_ID + "=" + id);
			if (dc==null || dc.getCount()==0) {
				return null;
			}
			Document doc = dc.getFirstDocument();
			return doc;
		} finally {
			trash.add(dc);
		}
	}
	
	/**
	 * Erzeugt ein neues Document mit dem Toy
	 * Da das Document doc nur intern in dieser Methode und nur temporr
	 * bentigt wird, kann die Methode selbst das Recycle bernehmen.
	 * @param id - ID des neuen Toys
	 * @param name - Name des neuen Toys
	 * @throws NotesException
	 */
	public void addToy (int id, String name) throws NotesException {
		checkRecycled();
		Document doc = null;
		try {
			doc = db.createDocument();
			doc.replaceItemValue (ITEM_ID, new Integer (id));
			doc.replaceItemValue (ITEM_NAME, name);
			doc.replaceItemValue (ITEM_FORM, DEFAULT_FORM);			
			doc.save();
		} finally {
			GC.recycle (doc);
		}
	}
	/**
	 * Lscht das Document des Toy mit der ID id.
	 * Da das Document doc nur intern in dieser Methode und nur temporr
	 * bentigt wird, kann die Methode selbst das Recycle bernehmen.
 	 * @param id
	 * @throws NotesException
	 */
	public void removeToy (int id) throws NotesException {
		checkRecycled();
		Document doc = null;
		try {
			doc = this.getToyDocument(id);
			if (doc!=null) {
				doc.remove(true);
			}
		} finally {
			doc.recycle();
		}
	}
	
	/**
	 * Da die Klasse Toy eine Instanzvariable Database db
	 * selbst als Objekt bezieht und vorhlt, muss eine
	 * externe Mglichkeit geboten werden, dieses Domino Java
	 * Objekt dem Recycle zuzufhren.
	 * Gleichzeitig wird ber diese Methode die
	 * Instanzvariable db auf null gesetzt, um in checkRecycled
	 * erkennen zu knnen, ob ein Recycling erfolgt war.
	 */
	public void recycle () {
		try {
			if (trash != null && trash.size() > 0) {
				session.recycle(trash);
			}
		} catch (NotesException e) {
			e.printStackTrace();
		}
		db = GC.recycle(db);
	}
	
	/**
	 * Prft, ob das ToyStore Objekt noch gltig ist.
	 * @throws NotesException
	 */
	private void checkRecycled () throws NotesException {
		if (db==null) {
			throw new NotesException(NotesError.NOTES_ERR_DELETED,
					"Object has been removed or recycled");
		}
	}
}
